/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  CrSpace.cpp - CR-SPACE representation class
 *
 *  Version: $Id: CrSpace.cpp 2752 2006-01-19 14:40:17Z mst $
 *
 */
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "CrSpace.h"
#include "Param.h"
#include "utils.h"

namespace std {}; using namespace std;

bool CrSpace::warn;
bool CrSpace::_par_debug;
bool CrSpace::_cr_debug;
bool CrSpace::_bl_debug;

////////////////////////////////////////////////////////////////////////
template <class _ForwardIterator>
static bool is_homogeneous(const _ForwardIterator& b, const _ForwardIterator& e)
{
    _ForwardIterator previ = b;
    for(_ForwardIterator i = b; i != e; ++i)
    {
        if (*previ != *i)
            return false;
    }
    return true;
} // is_homogeneous


////////////////////////////////////////////////////////////////////////
CrSpace::Chunk::Chunk(Param* par, const u_int32_t bitsize, u_int32_t value)
    : v(bitsize), p(par)
{
    for (u_int32_t i=0; i<bitsize; i++, value >>= 1)
        v[i] = value & 0x01;
} // CrSpace::Chunk::Chunk


////////////////////////////////////////////////////////////////////////
void CrSpace::set_debugs()
{
    _par_debug = getenv("MIC_PAR_DEBUG") ? true : false;
    _cr_debug = getenv("MIC_CR_DEBUG") ? true : false;
    _bl_debug = getenv("MIC_CRBLOCKS_DEBUG") ? true : false;
} // CrSpace::set_debugs


////////////////////////////////////////////////////////////////////////
bool CrSpace::add(Param* par, const u_int32_t bitoffs,
                  const u_int32_t bitsize, u_int32_t value)
{
    Chunk chunk(par, bitsize, value);

    //printf(CrSpace::add - "\"%s.%s\" 0x%x/%d /%d/ <= 0x%x /%c/\n",
    //       par->group->name.c_str(), par->name.c_str(),
    //       bitoffs, bitoffs, bitsize, value, isprint(value)?value:'.');
    
    #ifndef TAVOR_DEBUG 
    pair<Cr::iterator,bool> p=_cr.insert(pair<u_int32_t,Chunk>(bitoffs, chunk));
    #else
    pair<Cr::iterator,bool> p;
    p.second = true;
    _cr.push_back(pair<u_int32_t,Chunk>(bitoffs, chunk));
    #endif

    if (!p.second)
    {
        Chunk old = p.first->second;
        if (warn) {
            printf(
	              "-W- Parameter \"%s.%s\" %s:%d initializes same CR-SPACE location (0x%x.%d) "
                      "as parameter \"%s.%s\" %s:%d\n",
                      par->group->name.c_str(), par->name.c_str(),
                      par->fname.c_str(), par->line,
                      bitoffs >> 5, bitoffs & 0x1f,
                      old.p->group->name.c_str(), old.p->name.c_str(),
                      old.p->fname.c_str(), old.p->line);
        }
    }
    if (_par_debug)
    {
        printf("Par:%7s.%-35s; CrSpace 0x%06x.%02d /size:%02d/ <= 0x%8x. boot2=%d\n",
               par->group->name.c_str(), par->name.c_str(), ((bitoffs >> 5) << 2),
               bitoffs & 0x1f, bitsize, value, par->boot2);
    }

    return true;
} // CrSpace::add


////////////////////////////////////////////////////////////////////////
bool CrSpace::create_inits(bool only_rmw)
{

    #ifndef TAVOR_DEBUG
    #else
    sort(_cr.begin(), _cr.end(), SortAscending());
    #endif


    // Handle possible overlaps
    u_int32_t prev_offs = 0;
    u_int32_t prev_size = 0;
    Param     *prev_par = 0;
    for (Cr::iterator cr_it = _cr.begin(); cr_it != _cr.end(); ++cr_it)
    {
        Chunk     ch = cr_it->second;
        u_int32_t cur_offs = cr_it->first;
        u_int32_t cur_size = ch.v.size();

        if (cur_offs < prev_offs + prev_size)
            // TAVOR DEBUG- Remove!!! return errmsg(
            if (warn) {
                printf("-W- "
                       "Parameter \"%s.%s\" 0x%x.%d (%d)\n"
                       "                 %s:%d\n"
                       "       overlaps with \"%s.%s\" 0x%x.%d (%d)\n"
                       "                 %s:%d\n",
                       ch.p->group->name.c_str(), ch.p->name.c_str(),
                       cur_offs >> 5, cur_offs & 0x1f, cur_size,
                       ch.p->fname.c_str(), ch.p->line,
                       prev_par->group->name.c_str(), prev_par->name.c_str(),
                       prev_offs >> 5, prev_offs & 0x1f, prev_size,
                       prev_par->fname.c_str(), prev_par->line);
            }

        prev_offs = cur_offs;
        prev_size = cur_size;
        prev_par = ch.p;
    }


    // Now create initialization by words map
    typedef pair<u_int32_t, u_int32_t> WordMask;    // WordsValue/Mask
    #ifndef TAVOR_DEBUG
    typedef map<u_int32_t, WordMask>   Inits;       // Word_offs / (Value/Mask)
    #else
    typedef vector<pair<u_int32_t, WordMask> >   Inits;       // Word_offs / (Value/Mask)
    #endif
    
    Inits   inits;
    for (Cr::iterator cr_it = _cr.begin(); cr_it != _cr.end(); ++cr_it)
    {
        Chunk     ch = cr_it->second;
        u_int32_t bit_offs = cr_it->first & 0x1f;
        u_int32_t bit_size = ch.v.size();
        u_int32_t word_offs = (cr_it->first >> 5) << 2;

        // Create value (in a corrent position in a word)
        u_int32_t value = 0;
        for (u_int32_t i=0; i<bit_size; i++)
            value |= (ch.v[i] << i);
        value <<= bit_offs;

        // Now create mask
        // Remember, fields can't cross word boundary (checked in XML parser)
        u_int32_t mask = bit_offs ? (1<<bit_offs)-1 : 0;
        mask |= ((bit_offs+bit_size) == 32) ? 0 : ~((1<<(bit_offs+bit_size))-1);
        mask = ~mask;

        //printf("\"%s.%s\" - offs:0x%x.%d (%d), word_offs:0x%x mask:0x%x value:0x%x\n",
        //       ch.p->group->name.c_str(), ch.p->name.c_str(),
        //       cr_it->first >> 5, cr_it->first & 0x1f, bit_size, word_offs,
        //       mask, value);

        // Now insert if doesn't exist or merge if exists
	#ifndef TAVOR_DEBUG
	pair<Inits::iterator, Inits::iterator> ip = inits.equal_range(word_offs);
	#else
	pair<Inits::iterator, Inits::iterator> ip(inits.end(), inits.end());
	#endif
	
        if (ip.first == ip.second)
        {
            WordMask new_w(value, mask);
            inits.insert(ip.first, Inits::value_type(word_offs, new_w));
        }
        else
        {
            WordMask& oldw = ip.first->second;
            oldw.first  |= value;
            oldw.second |= mask;
        }
    }

    if (_cr_debug)
    {
        printf("CR-SPACE initialization:\n");
        for (Inits::iterator it = inits.begin(); it != inits.end(); ++it)
            printf("CRSPACE[0x%08x] = 0x%x (mask 0x%x)\n", it->first,
                   it->second.first, it->second.second);
        printf("\n");
    }

    // Now detect initialization fragments per init types
    u_int32_t         prev_addr = 0xffffffff;
    u_int32_t         addr_candidate=0;
    vector<u_int32_t> block_candidate;
    for (Inits::iterator it = inits.begin(); it != inits.end(); ++it)
    {
        u_int32_t addr  = it->first;
        u_int32_t value = it->second.first;
        u_int32_t mask  = it->second.second;

        if (mask != 0xffffffff || only_rmw)
        {
            // Generate appropriate init if necessary
            generate_block(addr_candidate, block_candidate);
            block_candidate.clear();

            // Read-modify write
            rmw_init.push_back(addr);
            rmw_init.push_back(value);
            rmw_init.push_back(mask);

            prev_addr = 0xffffffff;   // Can't continue block filling from here
        }
        else
        {
            if (addr == prev_addr + 4)
            {
                // continue to fill block_candidate
                block_candidate.push_back(value);
            }
            else
            {
                // Generate appropriate init if necessary
                generate_block(addr_candidate, block_candidate);

                // Now start new block candidate
                block_candidate.clear();
                addr_candidate = addr;
                block_candidate.push_back(value);
            }

            prev_addr = addr;
        }
    }
    // Finish last block
    generate_block(addr_candidate, block_candidate);

    if (_bl_debug)
    {
        printf("CR-SPACE blocks:\n");
        printf("----------------\n\n");

        printf("Individual words initialization - ");
        if (iwi_init.empty())
            printf("EMPTY\n");
        else
            printf("%ld dwords:\n", (long) iwi_init.size()/2);
        for (unsigned i=0; i<iwi_init.size(); i += 2)
            printf("    0x%08x <= 0x%08x\n", iwi_init[i], iwi_init[i+1]);
        printf("\n");

        printf("Homogeneous block initialization - ");
        if (hbi_init.empty())
            printf("EMPTY\n");
        else
            printf("\n");
        for (unsigned i=0; i<hbi_init.size(); i += 3)
            printf("    0x%08x <= %d dwords of 0x%08x\n",
                   hbi_init[i], hbi_init[i+1], hbi_init[i+2]);
        printf("\n");

        printf("Non homogeneous block initialization - ");
        if (nbi_init.empty())
            printf("EMPTY\n");
        else
            printf("\n");
        for (unsigned i=0; i<nbi_init.size(); )
        {
            unsigned cnt = nbi_init[i+1];
            printf("    0x%08x <= %d dwords of data\n", nbi_init[i], cnt);
            i += 2;
            for (unsigned j=0; j<cnt; j++, i++)
                printf("              0x%08x\n", nbi_init[i]);
        }
        printf("\n");

        printf("Read-modify-write block initialization - ");
        if (rmw_init.empty())
            printf("EMPTY\n");
        else
            printf("%ld dwords:\n", (long) rmw_init.size()/3);
        for (unsigned i=0; i<rmw_init.size(); i += 3)
            printf("    0x%08x <= 0x%08x; mask:0x%08x\n",
                   rmw_init[i], rmw_init[i+1], rmw_init[i+2]);
        printf("\n");
    }
    return true;
} // CrSpace::create_inits


////////////////////////////////////////////////////////////////////////
void CrSpace::generate_block(u_int32_t addr, vector<u_int32_t>& block)
{
    if (block.empty())
        return;

    if (block.size() < BLOCK_TRESHOLD)
    {
        // Block to small for block intialization
        // - Individual words initialization
        for (unsigned i=0; i<block.size(); ++i)
        {
            iwi_init.push_back(addr);
            iwi_init.push_back(block[i]);
            addr += 4;
        }
    }
    else if (is_homogeneous(block.begin(), block.end()))
    {
        // Homogeneous block initialization
        hbi_init.push_back(addr);
        hbi_init.push_back(block.size());
        hbi_init.push_back(block[0]);
    }
    else
    {
        // Non-homogeneous block initialization
        nbi_init.push_back(addr);
        nbi_init.push_back(block.size());
        for (unsigned i=0; i<block.size(); ++i)
            nbi_init.push_back(block[i]);
    }
} // CrSpace::generate_block
